/**
 * @file rpc_subscribe_example.c
 * @author Michal Vasko <mvasko@cesnet.cz>
 * @brief example of an application subscribing to an RPC
 *
 * @copyright
 * Copyright (c) 2019 CESNET, z.s.p.o.
 *
 * This source code is licensed under BSD 3-Clause License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://opensource.org/licenses/BSD-3-Clause
 */
#define _GNU_SOURCE
#define _XOPEN_SOURCE 500

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <inttypes.h>

#include <libyang/libyang.h>

#include "sysrepo.h"

volatile int exit_application = 0;
volatile unsigned int ue_prbusage_ul_switch = 0; // 0: stop, 1: start

static int
rpc_cb(sr_session_ctx_t *session, const char *path, const sr_val_t *input, const size_t input_cnt,
        sr_event_t event, uint32_t request_id, sr_val_t **output, size_t *output_cnt, void *private_data)
{
    int i;
    // 1. assert 
    for(i=0;i<input_cnt;i++){
        if(!strcmp(input[i].xpath,"/yzmm-rpc-notify:data-capture/dc-type")){
            printf("recv %s\n",input[i].data.enum_val);
        } else if(!strcmp(input[i].xpath, "/yzmm-rpc-notify:data-capture/dc-action")){            
            ue_prbusage_ul_switch = input[i].data.uint32_val; 
            printf("ue_prbusage_ul_switch: %d\n",ue_prbusage_ul_switch);
        }
    }

    // 2. fill output
    *output = malloc(sizeof **output);
    *output_cnt = 1;
    (*output)[0].xpath = strdup("/yzmm-rpc-notify:data-capture/dc-result");
    (*output)[0].type = SR_ENUM_T;
    (*output)[0].data.enum_val = strdup("OK");
    return SR_ERR_OK;
}

static void
sigint_handler(int signum)
{
    (void)signum;

    exit_application = 1;
}

int
main(int argc, char **argv)
{
    sr_conn_ctx_t *connection = NULL;
    sr_session_ctx_t *session = NULL;
    sr_subscription_ctx_t *subscription = NULL;
    int rc = SR_ERR_OK;

    /* turn logging on */
    sr_log_stderr(SR_LL_WRN);

    /* connect to sysrepo */
    rc = sr_connect(0, &connection);
    if (rc != SR_ERR_OK) {
        goto cleanup;
    }

    /* start session */
    rc = sr_session_start(connection, SR_DS_RUNNING, &session);
    if (rc != SR_ERR_OK) {
        goto cleanup;
    }

    /* subscribe for /yzmm-rpc-notify:data-capture */
    rc = sr_rpc_subscribe(session, "/yzmm-rpc-notify:data-capture", rpc_cb, NULL, 0, 0, &subscription);
    if (rc != SR_ERR_OK) {
        goto cleanup;
    }
    printf("subscribe success and waiting for rpc action\n");

    /* loop until ctrl-c is pressed / SIGINT is received */
    signal(SIGINT, sigint_handler);
    signal(SIGPIPE, SIG_IGN);
    while (!exit_application) {
        
        sleep(1000);
    }

    printf("Application exit requested, exiting.\n");

cleanup:
    sr_disconnect(connection);
    return rc ? EXIT_FAILURE : EXIT_SUCCESS;
}

